#include "TePDIHaralick.hpp"

#include <TeGeometryAlgorithms.h>
#include <TeCoord2D.h>
#include <TeAgnostic.h>

#include <math.h>

#define INSERT_CONCURRENCE( concpixel_line, concpixel_col ) \
  { \
    ponto_aux.y( concpixel_line ); \
    ponto_aux.x( concpixel_col ); \
    ponto_aux2 = input_raster_->index2Coord( ponto_aux ); \
    if( TeWithin( ponto_aux2, pol ) ) { \
      if( input_raster_->getElement( concpixel_col, concpixel_line, auxkey.second, \
        band ) ) { \
        auxkey.first = itp.operator*(band); \
        ccm_it = mat->find( auxkey ); \
        if( ccm_it == mat->end() ) { \
          (*mat)[ auxkey ] = 1.0; \
        } else { \
          ccm_it->second += 1.0; \
        } \
      } \
    } \
  };
  

TePDIHaralick::TePDIHaralick( int dirmask )
{
  polygonset_.reset( new TePolygonSet );
  dirmask_ = dirmask;
}


TePDIHaralick::~TePDIHaralick()
{
}


void TePDIHaralick::ResetState( const TePDIParameters& params )
{
  conc_matrix_cache_.clear();
  
  params.GetParameter( "input_raster", input_raster_ );
  params.GetParameter( "polygonset", polygonset_ );
}


bool TePDIHaralick::RunImplementation()
{
  TEAGN_LOG_AND_THROW( "This function cannot be used for this class" );
  return false;
}


bool TePDIHaralick::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster", 
    input_raster ),
    "Missing parameter: input_raster" );
  TEAGN_TRUE_OR_RETURN( input_raster.isActive(),
    "Invalid parameter: input_raster inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster not ready" );

  /* Checking the restriction polygon set */
  
  TePDITypes::TePDIPolygonSetPtrType polygonset;  
  
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "polygonset", polygonset ),
    "Missing parameter : polygonset" );
  
  TEAGN_TRUE_OR_RETURN( polygonset.isActive(), 
    "Invalid parameter : polygonset" );
    
  TEAGN_TRUE_OR_RETURN( polygonset->size() > 0, 
    "Invalid parameter : polygonset" );    
      
  return true;
}


bool TePDIHaralick::getEntropy( unsigned int band, unsigned int pol_index,
  double& value )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  ConcMatrixPtrT cmatrix = getConcurrenceMatrix( band, pol_index );
    
  if( cmatrix.isActive() ) {
    value = 0.0;
  
    ConcMatrixT::iterator itFirst = cmatrix->begin();
    ConcMatrixT::iterator itEnd = cmatrix->end();
    double ocurrences = 0;
    double entropia = 0;
  
    while ( itFirst != itEnd )
    {
      ocurrences = itFirst->second;
      
      //entropia += ( (ocurrences)* log(ocurrences) );
      entropia += ( ocurrences * ( log(ocurrences) / log( (double)10.0 ) ) );
      
      ++itFirst;
    }
  
    value = ( -1 ) * entropia;
    
    return true;
  } else {
    return false;
  }
}



bool TePDIHaralick::getEnergy( unsigned int band, unsigned int pol_index,
  double& energyValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  ConcMatrixPtrT cmatrix = getConcurrenceMatrix( band, pol_index );
    
  if( cmatrix.isActive() ) 
  {
    energyValue = 0.0;
  
    ConcMatrixT::iterator itFirst = cmatrix->begin();
    ConcMatrixT::iterator itEnd = cmatrix->end();
      
    while ( itFirst != itEnd )
    {
      energyValue += (pow ((itFirst->second),2));
    ++itFirst;
    }
    
    return true;
  }
  else 
  {
    return false;
  }
}


bool TePDIHaralick::getContrast( unsigned int band, unsigned int pol_index,
  double& contrastValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  ConcMatrixPtrT cmatrix = getConcurrenceMatrix( band, pol_index );
    
  if( cmatrix.isActive() ) 
  {
    contrastValue = 0.0;
  
  // The auxValue stores the probability of pair of level of gray i, j to occur.
  // The dif stores difference i - j.
  double auxValue, dif;
  
    ConcMatrixT::iterator itFirst = cmatrix->begin();
    ConcMatrixT::iterator itEnd = cmatrix->end();
      
    while ( itFirst != itEnd )
    {
    auxValue = itFirst->second;
    dif = ((itFirst->first.first) - (itFirst->first.second));
    contrastValue += ((pow(dif ,2)) * (auxValue)); 
    ++itFirst;
    }
    
    return true;
  }
  else 
  {
    return false;
  }
}


bool TePDIHaralick::getHomogeneity( unsigned int band, unsigned int pol_index,
  double& homogeneityValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  ConcMatrixPtrT cmatrix = getConcurrenceMatrix( band, pol_index );
    
  if( cmatrix.isActive() ) 
  {
    homogeneityValue = 0.0;
  
  // The dif stores difference i - j.
  double dif;
  
    ConcMatrixT::iterator itFirst = cmatrix->begin();
    ConcMatrixT::iterator itEnd = cmatrix->end();
      
    while ( itFirst != itEnd )
    {
    dif = ((itFirst->first.first) - (itFirst->first.second));
    homogeneityValue += ((itFirst->second)/(1+(pow(dif,2))));
    ++itFirst;
    }
    
    return true;
  }
  else 
  {
    return false;
  }
}


bool TePDIHaralick::getQuiSquare( unsigned int band, unsigned int pol_index,
  double& QuiSquareValue )
{
  TEAGN_TRUE_OR_THROW( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_TRUE_OR_THROW( pol_index < polygonset_->size(), 
    "Invalid polygon index" );

  ConcMatrixPtrT cmatrix = getConcurrenceMatrix( band, pol_index );
    
  if( cmatrix.isActive() ) 
  {
    QuiSquareValue = 0.0;

  // total_col stores the sum of the probabilities of column j
  // total_ln stores the sum of the probabilities of line i
  double total_col =  0.0, total_ln = 0.0;
  
    ConcMatrixT::iterator itFirst = cmatrix->begin();
    ConcMatrixT::iterator itEnd = cmatrix->end();

  //iterators and map auxiliaries
    ConcMatrixT::iterator itAux1 = cmatrix->begin();
      
  map <double, double> totalLine;
  map <double, double> totalColumn;

  while(itAux1 != itEnd)
  { 
    if (totalLine.find(itAux1->first.first)== totalLine.end())
    {
      totalLine[itAux1->first.first] = itAux1->second;
    }
    else
    {
      totalLine[itAux1->first.first] += itAux1->second;
    }

    if (totalColumn.find(itAux1->first.second) == totalColumn.end())
    {
      totalColumn[itAux1->first.second] = itAux1->second;
    }
    else
    {
      totalColumn[itAux1->first.second] += itAux1->second;
    }

    ++itAux1;

  }

  
  while ( itFirst != itEnd )
    {
    total_col = (totalColumn.find(itFirst->first.second))->second;
    total_ln = (totalLine.find(itFirst->first.first))->second;

    QuiSquareValue += ((pow(itFirst->second,2)))/( total_col * total_ln);
    ++itFirst;
  }

    return true;
  }
  else 
  {
    return false;
  }
}


TePDIHaralick::ConcMatrixPtrT TePDIHaralick::getConcurrenceMatrix( 
  unsigned int band, unsigned int pol_index )
{
  TEAGN_DEBUG_CONDITION( ( ((int)band) < input_raster_->params().nBands() ),
    "Invalid raster band" );
  TEAGN_DEBUG_CONDITION( ( pol_index < polygonset_->size() ), 
    "Invalid polygon index" );
    
  ConcMatrixCacheKeyT key;
  key.first = band;
  key.second = pol_index;
  
  ConcMatrixCacheT::iterator it = conc_matrix_cache_.find( key );
  
  if( it == conc_matrix_cache_.end() ) {
    TePolygon& pol = (*polygonset_)[ pol_index ];
  
    //itp -> pixel sobre o qual o clculo est sendo efetuado
    TeRaster::iteratorPoly itp = input_raster_->begin(pol,TeBoxPixelIn, band);
 
    //declare the map that will be the cooccurrence matriz
    ConcMatrixPtrT mat( new ConcMatrixT );

    TeCoord2D ponto_aux;
    TeCoord2D ponto_aux2;
    ConcMatrixKeyT auxkey;
  
    ConcMatrixT::iterator ccm_it;
    
    int curr_line = 0;
    int curr_col = 0;
    
    //enquanto no acabar os pixels do polgono n
    while ( ! itp.end() ) {
      curr_line =  itp.currentLine();
      curr_col = itp.currentColumn();
    
      if( dirmask_ & North ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col )
      if( dirmask_ & NorthEast ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col + 1 )
      if( dirmask_ & East ) 
        INSERT_CONCURRENCE( curr_line, curr_col + 1 )
      if( dirmask_ & SouthEast ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col + 1 )        
      if( dirmask_ & South ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col )         
      if( dirmask_ & SouthWest ) 
        INSERT_CONCURRENCE( curr_line + 1, curr_col - 1 )        
      if( dirmask_ & West ) 
        INSERT_CONCURRENCE( curr_line, curr_col - 1 )         
      if( dirmask_ & NorthWest ) 
        INSERT_CONCURRENCE( curr_line - 1, curr_col - 1 )           
    
      ++itp;
    } 
      
    /* Normalizing matrix values */
      
    ccm_it = mat->begin();
    ConcMatrixT::iterator ccm_it_end = mat->end();
      
    double total = 0;
      
    while ( ccm_it != ccm_it_end )
    {
      total += ccm_it->second;
        
      ccm_it++;
    }      
      
    ccm_it = mat->begin();
    ccm_it_end = mat->end();
      
    while ( ccm_it != ccm_it_end )
    {
      ccm_it->second = ccm_it->second / total;
        
      ++ccm_it;
    }
      
    /* Updading concurrence matrix cache */
      
    conc_matrix_cache_[ key ] = mat;

    return mat;  
  } else {
    return it->second;
  }
}


