 
#include "TePDIIHSFusionStrategy.hpp"
#include "TePDIUtils.hpp"
#include <TeAgnostic.h>
#include "TePDIColorTransform.hpp"
#include "TePDIAlgorithmFactory.hpp"
#include "TePDIPIManager.hpp"
#include "TePDIStatistic.hpp"
#include "TePDIContrast.hpp"

#include <TeRasterRemap.h>


TePDIIHSFusionStrategy::TePDIIHSFusionStrategy()
{
};      


TePDIIHSFusionStrategy::~TePDIIHSFusionStrategy()
{
};


bool TePDIIHSFusionStrategy::CheckParameters( 
  const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType reference_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "reference_raster", 
    reference_raster ),
    "Missing parameter: reference_raster" );
  TEAGN_TRUE_OR_RETURN( reference_raster.isActive(),
    "Invalid parameter: reference_raster inactive" );
  TEAGN_TRUE_OR_RETURN( reference_raster->params().status_ != 
    TeRasterParams::TeNotReady, 
    "Invalid parameter: reference_raster not ready" );
    
  /* Checking input_raster2 */
  
  TePDITypes::TePDIRasterPtrType lowres_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "lowres_raster", 
    lowres_raster ),
    "Missing parameter: lowres_raster" );
  TEAGN_TRUE_OR_RETURN( lowres_raster.isActive(),
    "Invalid parameter: lowres_raster inactive" );
  TEAGN_TRUE_OR_RETURN( lowres_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: lowres_raster not ready" );    
  TEAGN_TRUE_OR_RETURN( ( lowres_raster->nBands() == 3 ),
    "Invalid parameter: lowres_raster must have 3 bands ( RGB )" );       
    
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );    
    
  /* Checking bands */    
    
  int reference_raster_band = 0;    
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "reference_raster_band", 
    reference_raster_band ), "Missing parameter : reference_raster_band" );
  TEAGN_TRUE_OR_RETURN( ( reference_raster->nBands() > reference_raster_band ),
    "Invalid parameter : reference_raster_band" );
    
  /* Checking photometric interpretation */
  
  TEAGN_TRUE_OR_RETURN( ( 
    ( reference_raster->params().photometric_[ reference_raster_band ] == 
      TeRasterParams::TeRGB ) ||
    ( reference_raster->params().photometric_[ reference_raster_band ] == 
      TeRasterParams::TeMultiBand ) ),
  "Invalid parameter - reference_raster (invalid photometric interpretation)" );   

  for( int lowres_raster_band = 0 ; 
    lowres_raster_band < lowres_raster->params().nBands() ; 
    ++lowres_raster_band ) {
    
    TEAGN_TRUE_OR_RETURN( ( 
      ( lowres_raster->params().photometric_[ lowres_raster_band ] == 
        TeRasterParams::TeRGB ) ||
      ( lowres_raster->params().photometric_[ lowres_raster_band ] == 
        TeRasterParams::TeMultiBand ) ),
    "Invalid parameter - lowres_raster (invalid photometric interpretation)" ); 
  }  
    
  return true;
}


bool TePDIIHSFusionStrategy::Implementation( const TePDIParameters& params )
{
  TePDITypes::TePDIRasterPtrType reference_raster;
  params.GetParameter( "reference_raster", reference_raster );
  
  TePDITypes::TePDIRasterPtrType lowres_raster;
  params.GetParameter( "lowres_raster", lowres_raster );
  
  TePDITypes::TePDIRasterPtrType output_raster;
  params.GetParameter( "output_raster", output_raster );
    
  int reference_raster_band = 0;    
  params.GetParameter( "reference_raster_band", reference_raster_band );
  
  /* Bringing lowres_raster to reference_raster size */
   
  TePDITypes::TePDIRasterPtrType upsampled_raster;
  
  {
    TeRasterParams upsampled_raster_params = lowres_raster->params();
    upsampled_raster_params.setNLinesNColumns( 1, 1 );
    
    TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( upsampled_raster,
      upsampled_raster_params, TePDIUtils::TePDIUtilsAutoMemPol ),
      "Unable create the new upsampled raster RGB raster" );
      
	  TEAGN_TRUE_OR_RETURN( TePDIUtils::resampleRasterByLinsCols(
	    lowres_raster, 
		  upsampled_raster, 
		  (unsigned int)reference_raster->params().nlines_, 
		  (unsigned int)reference_raster->params().ncols_, 
		  progress_interface_enabled_,
		  TePDIInterpolator::NNMethod), "Raster resample error" );    
/*      
TEAGN_TRUE_OR_THROW( TePDIUtils::TeRaster2Geotiff( upsampled_raster,
  "TePDIIHSFusionStrategy_upsampled_raster.tif" ), 
  "GeoTIF generation error" );     
*/      
  }
   
  /* Generating the IHS raster */
  
  TePDITypes::TePDIRasterPtrType ihs_raster;
  TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( ihs_raster,
    1, 10, 10, false, TeFLOAT, 0 ),
    "Unable create the new ihs_raster raster RGB raster" );  
    
  double rgb_channels_min = 0;
  double rgb_channels_max = 0;
  TEAGN_TRUE_OR_RETURN( TePDIUtils::TeGetRasterMinMaxBounds( 
    upsampled_raster,
    0, rgb_channels_min, rgb_channels_max ),
    "Unable to get channel range" )
     
  {
    TePDIParameters rgb2ihs_params;
    
    TePDITypes::TePDIRasterVectorType input_rasters;
    input_rasters.push_back( upsampled_raster );
    input_rasters.push_back( upsampled_raster );
    input_rasters.push_back( upsampled_raster );
    rgb2ihs_params.SetParameter( "input_rasters", input_rasters );
    
    std::vector< int > input_channels;
    input_channels.push_back( 0 );
    input_channels.push_back( 1 );
    input_channels.push_back( 2 );
    rgb2ihs_params.SetParameter( "input_channels", input_channels );
    
    TePDITypes::TePDIRasterVectorType output_rasters;
    output_rasters.push_back( ihs_raster );
    rgb2ihs_params.SetParameter( "output_rasters", output_rasters );
    
    rgb2ihs_params.SetParameter( "transf_type", TePDIColorTransform::Rgb2Ihs );  

    rgb2ihs_params.SetParameter( "rgb_channels_min", rgb_channels_min );
    rgb2ihs_params.SetParameter( "rgb_channels_max", rgb_channels_max );    
    
    TePDIColorTransform transform_instance;
    TEAGN_TRUE_OR_RETURN( transform_instance.Reset( rgb2ihs_params ),
      "Error in IHS parameters" )
      
    transform_instance.ToggleProgInt( progress_interface_enabled_ );
      
    TEAGN_TRUE_OR_RETURN( transform_instance.Apply(), 
      "Unable to build IHS color space" );
  }
  
/*  
TEAGN_TRUE_OR_THROW( TePDIUtils::TeRaster2Geotiff( ihs_raster,
  "TePDIIHSFusionStrategy_ihs_raster.tif" ), 
  "GeoTIF generation error" );    
*/

  /* Free the anused raster */
  
  upsampled_raster.reset();  
  
  /* Bring the reference raster to the mean and variance of
     the I channel */
  
  TePDITypes::TePDIRasterPtrType ref_rast_contrast;
  
  {
    {
      /* Get target stat params */
      
      TePDIParameters statparams;
      
      TePDITypes::TePDIRasterVectorType rasters;
      rasters.push_back( ihs_raster );
      statparams.SetParameter( "rasters", rasters );
      
      std::vector< int > bands;
      bands.push_back( 0 ); // IHS raster I band
      statparams.SetParameter( "bands", bands );
      
      TePDIStatistic stat;
      stat.ToggleProgInt( progress_interface_enabled_ );
      TEAGN_TRUE_OR_THROW( stat.Reset( statparams ),
        "Invalid statistic algorithm parameters" )
        
      double target_mean = stat.getMean( 0 );
      double target_variance = stat.getVariance( 0 );

      /* Applying contrast to the reference raster */
      
      TePDIParameters contparams;
      
      contparams.SetParameter( "contrast_type", 
        TePDIContrast::TePDIContrastStat );
      contparams.SetParameter( "input_image", reference_raster );
      contparams.SetParameter( "input_band", (int)reference_raster_band );
      contparams.SetParameter( "target_mean", target_mean );
      contparams.SetParameter( "target_variance", target_variance );
      
      TeRasterParams ref_rast_cont_params = reference_raster->params();
      ref_rast_cont_params.setNLinesNColumns( 1, 1 );
      ref_rast_cont_params.nBands( 1 );
      ref_rast_cont_params.setDataType( TeFLOAT, -1 );
      TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( ref_rast_contrast,
        ref_rast_cont_params, TePDIUtils::TePDIUtilsAutoMemPol ),
        "Unable create the new reference raster (contrast)" );  
      contparams.SetParameter( "output_image", ref_rast_contrast );
      
      contparams.SetParameter( "output_band", (int)0 );
      
      TePDIContrast contra;
      TEAGN_TRUE_OR_RETURN( contra.Apply( contparams ),
        "Error applying contrast" )
    }
  
  }
    
  /* Swapping reference_raster into ihs_raster I component */
  {
    const unsigned int lines_bound = ihs_raster->params().nlines_;
    const unsigned int cols_bound = ihs_raster->params().ncols_;
    unsigned int line = 0;
    unsigned int col = 0;
    TeRaster* inraster = ref_rast_contrast.nakedPointer();
    TeRaster* outraster = ihs_raster.nakedPointer();
    double value = 0;
    
    TePDIPIManager progress( "Swapping Intensity channel...", lines_bound,
      progress_interface_enabled_ );
    
    for( line = 0; line < lines_bound ; ++line ) {
      TEAGN_FALSE_OR_RETURN( progress.Increment(), "Canceled by the user" );
    
      for( col = 0; col < cols_bound ; ++col ) {
        inraster->getElement( col, line, value, 0 );
        
        outraster->setElement( col, line, value, 0 );
      }
    }
    
    progress.Toggle( false );
  }
  
/* 
TEAGN_TRUE_OR_THROW( TePDIUtils::TeRaster2Geotiff( ihs_raster,
  "TePDIIHSFusionStrategy_ihs_raster_swapped.tif" ), 
  "GeoTIF generation error" );     
*/  

  /* Switching back to RBG space */
  {
    TePDIParameters ihs2rgb_params;
    
    TePDITypes::TePDIRasterVectorType input_rasters;
    input_rasters.push_back( ihs_raster );
    input_rasters.push_back( ihs_raster );
    input_rasters.push_back( ihs_raster );
    ihs2rgb_params.SetParameter( "input_rasters", input_rasters );
    
    std::vector< int >input_channels;
    input_channels.push_back( 0 );
    input_channels.push_back( 1 );
    input_channels.push_back( 2 );
    ihs2rgb_params.SetParameter( "input_channels", input_channels );
    
    TePDITypes::TePDIRasterVectorType output_rasters;
    output_rasters.push_back( output_raster );    
    ihs2rgb_params.SetParameter( "output_rasters", output_rasters );
    
    ihs2rgb_params.SetParameter( "transf_type", TePDIColorTransform::Ihs2Rgb );  
    
    ihs2rgb_params.SetParameter( "rgb_channels_min", rgb_channels_min );
    ihs2rgb_params.SetParameter( "rgb_channels_max", rgb_channels_max );
    
    TePDIColorTransform transform_instance;
    TEAGN_TRUE_OR_RETURN( transform_instance.Reset( ihs2rgb_params ),
      "Error in IHS parameters" )    
    
    TEAGN_TRUE_OR_RETURN( transform_instance.Apply(), 
      "Unable to build RGB color space" );      
  }
   
  return true;
}

