/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIMATRIX_HPP
  #define TEPDIMATRIX_HPP
  
  #include <TeAgnostic.h>
  #include <TeSharedPtr.h>
  #include <TeMappedMemory.h>
  
  #include <TeUtils.h>
  
  #include <vector>
  

  /**
   * @brief This is the template class to deal with a generic matrix.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   * @ingroup PDIAux
   */
  template< class T >
  class TePDIMatrix {
    public :
      /** @typedef TeSharedPtr< TePDIMatrix< T > > pointer 
          Type definition for a instance pointer */
      typedef TeSharedPtr< TePDIMatrix< T > > pointer;
      
      /**
       * @brief Memory polycy.
       */
      enum MemoryPolicy {
        /**
         * Automatic memory policy ( Try to use RAM or DISK, if there is no 
         * avaliable RAM ) -
         * DO NOT USE AutoMemPol FOR COMPLEX DATA TYPES !
         * 
         */
        AutoMemPol,
        /**
         * RAM memory policy.
         */
        RAMMemPol,
        /**
         * Disk memory policy ( virtual mapped memory ) -
         *  DO NOT USE DiskMemPol FOR COMPLEX DATA TYPES !
         */
        DiskMemPol
      };
      
      /**
       * @brief Default Constructor.
       * @note The default mamory policy is RAMMemPol.
       */
      TePDIMatrix();      
      
      /**
       * @brief Alternative Constructor.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @param mp Memory policy.
       */
      TePDIMatrix( unsigned int lines, unsigned int columns, 
        MemoryPolicy mp );
        
      /**
       * @brief Alternative Constructor.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @note The default mamory policy is RAMMemPol.
       */
      TePDIMatrix( unsigned int lines, unsigned int columns );        

      /**
       * @brief Alternative Constructor.
       *
       * @param external External object reference.
       * @param mp Memory policy.
       */
      TePDIMatrix( const TePDIMatrix< T >& external );

      /**
       * @brief Default Destructor
       */
      ~TePDIMatrix();
      
      /**
       * @brief Reset the active instance the the new parameters.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @return true if OK, false on error.
       */
      bool Reset( unsigned int lines = 0, 
        unsigned int columns = 0 );      

      /**
       * @brief Reset the active instance the the new parameters.
       *
       * @param lines Number of lines.
       * @param columns Number of columns.
       * @param mp Memory policy.
       * @return true if OK, false on error.
       */
      bool Reset( unsigned int lines, unsigned int columns,
        MemoryPolicy mp );
      
      /**
       * @brief The number of current matrix lines.
       *
       * @return The number of current matrix lines.
       */
      inline unsigned int GetLines() const;
      
      /**
       * @brief The number of current matrix columns.
       *
       * @return The number of current matrix columns
       */
      inline unsigned int GetColumns() const;
      
      /**
       * @brief Empty Matrix verification.
       *
       * @return true if the matrix is empty.
       */
      inline bool IsEmpty() const;

      /**
       * @brief Operator = overload.
       *
       * @note The external memory policy will be used as reference.
       *
       * @param external External instance reference.
       * @return A reference to the current matrix.
       */
      const TePDIMatrix< T >& operator=( 
        const TePDIMatrix< T >& external );

      /**
       * @brief Operator () overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A reference to the required element.
       */
      inline T& operator()( const unsigned int& line, 
        const unsigned int& column );
      
      /**
       * @brief Operator () overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A const reference to the required element.
       */
      inline const T& operator()( const unsigned int& line, 
        const unsigned int& column ) const; 
      
      /**
       * @brief Operator [] overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs.
       */
      inline T* operator[]( const unsigned int& line );
      
      /**
       * @brief Operator [] overload.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs.
       */
      inline T const* operator[]( const unsigned int& line ) const;      
            
    protected :
    
      /**
       * @brief Mapped memory vectors node.
       */    
      class MMObjsVectorNodeT 
      {
        public:
        
          TeMappedMemory* mm_ptr_;
          
          unsigned int first_line_idx;
          
          unsigned int lines_bound_;
      };
      
      /**
       * @brief Mapped memory vectors node pionter.
       */        
      typedef MMObjsVectorNodeT* MMObjsVectorNodePtrT;
      
      /**
       * @brief Max bytes per mapped memory file.
       */
      mutable unsigned long int max_bytes_per_mapped_file_;      

      /**
       * @brief The lines pointers vector (RAM + mapped files).
       */
      mutable T** lines_pointers_vector_;
      
      /**
       * @brief The RAM lines pointers vector.
       * @note The mapped memory lines will have zero 
       * pointers.
       */
      mutable T** ram_lines_pointers_vector_;      
      
      /**
       * @brief The total lines number.
       */
      mutable unsigned int total_lines_;
      
      /**
       * @brief The total columns number.
       */
      mutable unsigned int total_columns_;     
      
      /**
       * @brief The current used memory policy.
       */
      mutable MemoryPolicy current_mem_policy_;
      
      /**
       * @brief The internal mapped memory handlers vector.
       */
      mutable std::vector< MMObjsVectorNodePtrT > 
        mapped_memory_objs_ptrs_vect_;
        
      /**
       * @brief A vector mapping each line to the
       * respective mapped memory object.
       */
      mutable MMObjsVectorNodePtrT* lines2mm_objs_vector_;
      
      /**
       * @brief A pointer to the current active
       * mapped memory object.
       */
      mutable MMObjsVectorNodePtrT current_active_mmobj_ptr_;
     
      /**
       * @brief Reset the internal variables to the initial state.
       */      
      void init();
      
      /**
       * @brief Clear all allocated resources and go back to the initial
       * state.
       */      
      void clear();      
      
      /**
       * @Allocate maped memory lines.
       * @param starting_line_index Starting line index.
       * @return true if OK, false on errors.
       */      
      bool allocateMMLines( unsigned int starting_line_index );
      
      /**
       * @brief Returns a pointer to the required line.
       *
       * @param line Line number.
       * @param column Column number.
       * @return A pointer to the required line.
       * @note The returned pointer is garanteed to 
       * be valid until an acess to another line occurs
       */
      T* scanLine( const unsigned int& line ) const;  
  };

  template< class T >
  void TePDIMatrix< T >::init()
  {
    max_bytes_per_mapped_file_ = ( 1024 * 1024 * 10 );
    lines_pointers_vector_ = 0;
    ram_lines_pointers_vector_ = 0;
    total_lines_ = 0;
    total_columns_ = 0;  
    current_mem_policy_ = RAMMemPol;
    lines2mm_objs_vector_ = 0;
    current_active_mmobj_ptr_ = 0;
  }
  

  template< class T >
  void TePDIMatrix< T >::clear()
  {
    if( lines_pointers_vector_ ) {
      delete[] lines_pointers_vector_;
    }
    
    if( ram_lines_pointers_vector_ ) {
      for( unsigned int curr_line = 0 ; curr_line < total_lines_ ;
        ++curr_line ) {
       
        if( ram_lines_pointers_vector_[ curr_line ] ) {
          delete[] ram_lines_pointers_vector_[ curr_line ];
        }
      }
      
      delete[] ram_lines_pointers_vector_;
    }

    for( unsigned int mmv_idx = 0 ; 
      mmv_idx < mapped_memory_objs_ptrs_vect_.size() ;
      ++mmv_idx ) {
      
      delete mapped_memory_objs_ptrs_vect_[ mmv_idx ]->mm_ptr_;
      delete mapped_memory_objs_ptrs_vect_[ mmv_idx ];
    }
    
    mapped_memory_objs_ptrs_vect_.clear();
    
    if( lines2mm_objs_vector_ ) {
      delete[] lines2mm_objs_vector_;
    }
  
    init();
  }  
  

  template< class T >
  TePDIMatrix< T >::TePDIMatrix()
  {
    init();
  }    
  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( unsigned int lines, 
    unsigned int columns, MemoryPolicy mp )
  {
    init();
    
    TEAGN_TRUE_OR_THROW( Reset( lines, columns, mp ),
      "Unable to initiate the matrix object" );
  }  
  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( unsigned int lines, 
    unsigned int columns )
  {
    init();
    
    TEAGN_TRUE_OR_THROW( Reset( lines, columns, RAMMemPol ),
      "Unable to initiate the matrix object" );
  }   
  
  
  template< class T >
  TePDIMatrix< T >::TePDIMatrix( const TePDIMatrix< T >& external )
  {
    init();
    
    operator=( external );
  }


  template< class T >
    TePDIMatrix< T >::~TePDIMatrix()
  {
    clear();
  }

  
  template< class T >
  bool TePDIMatrix< T >::Reset( unsigned int lines, 
    unsigned int columns )
  {
    return Reset( lines, columns, RAMMemPol );
  }  

  template< class T >
  bool TePDIMatrix< T >::Reset( unsigned int lines, 
    unsigned int columns,
    MemoryPolicy mp )
  {
    /* Update the old buffer if necessary */
    
    if( ( lines != total_lines_ ) || ( columns != total_columns_ ) ||
        ( current_mem_policy_ != mp ) ) {
    
      /* free the old resources */
      
      clear();
    
      /* Allocate the new resources */
      
      total_lines_ = lines;
      total_columns_ = columns;          
      
      current_mem_policy_ = mp;
      
      if( ( lines != 0 ) && ( columns != 0 ) ) {
        /* Guessing the memory source, if in automatic mode */
     
        unsigned int line = 0;
        
        /* Allocating the main lines pointers vectors */
        
        try
        {
          lines_pointers_vector_ = new T*[ total_lines_ ];
        }
        catch(...)
        {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        if( lines_pointers_vector_ == 0 ) {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        for( line = 0 ; line < total_lines_ ; ++line ) {
          lines_pointers_vector_[ line ] = 0;
        }
        
        try
        {
          ram_lines_pointers_vector_ = new T*[ total_lines_ ];
        }
        catch(...)
        {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        if( ram_lines_pointers_vector_ == 0 ) {
          clear();
          TEAGN_LOG_AND_RETURN( "Memory allocation error" );
        }
        
        for( line = 0 ; line < total_lines_ ; ++line ) {
          ram_lines_pointers_vector_[ line ] = 0;
        }

        /* Allocating lines */

        switch( current_mem_policy_ ) {
          case RAMMemPol :
          {
            T* new_line_ptr = 0;
        
            for( line = 0 ; line < total_lines_ ; ++line ) {
              try {
                new_line_ptr = new T[ total_columns_ ];
              }
              catch(...) {
                clear();
                
                TEAGN_LOG_AND_RETURN( "Memory allocation error" );               
              }
              
              if( new_line_ptr == 0 ) {
                clear();
                
                TEAGN_LOG_AND_RETURN( "Memory allocation error" );              
              } 
              
              lines_pointers_vector_[ line ] = new_line_ptr;
              ram_lines_pointers_vector_[ line ] = new_line_ptr;
            }
            
            break;
          }
          case DiskMemPol :
          {
            if( ! allocateMMLines( 0 ) ) {
              clear();
              
              TEAGN_LOG_AND_RETURN( 
                "Error allocating mapped memory lines" )
            }
            
            /* Forcing the first mapped memory tile to be 
                enabled */
            operator[]( 0 );               
              
            break;
          }       
          case AutoMemPol :
          {
            const unsigned long int free_vm = TeGetFreeVirtualMemory();
            
            if( free_vm < max_bytes_per_mapped_file_ )
            {
              clear();
              
              TEAGN_LOG_AND_RETURN( 
                "Error allocating mapped memory lines" )
            }
            
            const unsigned long int max_ram  = (unsigned long int)
              ( 0.75 * ( (double)(
              free_vm - max_bytes_per_mapped_file_ ) ) );    
            
            unsigned long int line_bytes = sizeof( T ) * columns;
            unsigned long int max_ram_lines = (unsigned long int)
              ( ( (double)max_ram ) / ( (double) line_bytes ) ); 
              
            T* new_line_ptr = 0;
        
            for( line = 0 ; line < total_lines_ ; ++line ) {
              if( line < max_ram_lines ) {
                try {
                  new_line_ptr = new T[ total_columns_ ];
                }
                catch(...) {
                  if( allocateMMLines( line ) ) {
                    /* Forcing the first mapped memory tile to be 
                        enabled */
                    operator[]( line );
                    
                    return true;
                  } else {
                    clear();
                    
                    TEAGN_LOG_AND_RETURN( 
                      "Error allocating mapped memory lines" )
                  }
                }
                
                if( new_line_ptr == 0 ) {
                  if( allocateMMLines( line ) ) {
                    /* Forcing the first mapped memory tile to be 
                        enabled */
                    operator[]( line );
                                      
                    return true;
                  } else {
                    clear();
                    
                    TEAGN_LOG_AND_RETURN( 
                      "Error allocating mapped memory lines" )
                  }           
                } 
                
                lines_pointers_vector_[ line ] = new_line_ptr;
                ram_lines_pointers_vector_[ line ] = new_line_ptr;
              } else { // ( line >= max_ram_lines )
                if( allocateMMLines( line ) ) {
                  /* Forcing the first mapped memory tile to be 
                      enabled */
                  operator[]( line );
                                    
                  return true;
                } else {
                  clear();
                  
                  TEAGN_LOG_AND_RETURN( 
                    "Error allocating mapped memory lines" )
                }              
              }
            }
            
            break;
          }                 
          default :
          {
            TEAGN_LOG_AND_THROW( "Invalid memory policy" );
            break;
          }
        }
      }
    }
    
    return true;
  }
  
  
  template< class T >
  inline unsigned int TePDIMatrix< T >::GetLines() const
  {
    return total_lines_;
  }

  
  template< class T >
  inline unsigned int TePDIMatrix< T >::GetColumns() const
  {
    return total_columns_;
  }
  
  
  template< class T >
  inline bool TePDIMatrix< T >::IsEmpty() const
  {
    return ( lines_pointers_vector_ == 0 ) ? true : false;
  }
  

  template< class T >
  const TePDIMatrix< T >& TePDIMatrix< T >::operator=(
    const TePDIMatrix< T >& external )
  {
    TEAGN_TRUE_OR_THROW( 
      Reset( external.total_lines_, external.total_columns_,
      external.current_mem_policy_ ),
      "Unable to initiate the matrix object" );
    
    unsigned int line;
    unsigned int column;
    
    for( line = 0 ; line < total_lines_ ; ++line ) {
      for( column = 0 ; column < total_columns_ ; ++column ) {
        operator()( line, column ) = external( line, column );
      }
    }

    return *this;
  }

  
  template< class T >
  inline T& TePDIMatrix< T >::operator()( const unsigned int& line, 
    const unsigned int& column )
  {
    TEAGN_DEBUG_CONDITION( ( line < total_lines_ ),
      "Invalid line" )
    TEAGN_DEBUG_CONDITION( ( column < total_columns_ ),
      "Invalid columns" )
      
    return scanLine( line )[ column ];
  }    
  

  template< class T >
  inline const T& TePDIMatrix< T >::operator()( const unsigned int& line, 
    const unsigned int& column ) const
  {
    TEAGN_DEBUG_CONDITION( ( line < total_lines_ ),
      "Invalid line" )
    TEAGN_DEBUG_CONDITION( ( column < total_columns_ ),
      "Invalid columns" )
        
    return scanLine( line )[ column ];
  }
  
  
  template< class T >
  inline T* TePDIMatrix< T >::operator[]( const unsigned int& line )
  {
    TEAGN_DEBUG_CONDITION( ( line < total_lines_ ),
      "Invalid line" )
  
    return scanLine( line );
  }    

 
  template< class T >
  inline T const* TePDIMatrix< T >::operator[]( const unsigned int& line ) 
    const
  {
    TEAGN_DEBUG_CONDITION( ( line < total_lines_ ),
      "Invalid line" )
  
    return scanLine( line );
  } 
  
  
  template< class T >
  bool TePDIMatrix< T >::allocateMMLines( 
    unsigned int starting_line_index )
  {
    TEAGN_DEBUG_CONDITION( lines_pointers_vector_ != 0, 
      "Trying to access an empty matrix" );
    TEAGN_DEBUG_CONDITION( total_columns_ != 0, 
      "Invalid number of columns" );      
    TEAGN_DEBUG_CONDITION( starting_line_index < total_lines_, 
      "Invalid starting_line_index" );
      
    /* Guessing the number of files */

    const unsigned long int bytes_per_line = total_columns_ * 
      sizeof( T );      
    const unsigned long int lines_per_file = (unsigned long int)
      ( ( (double) max_bytes_per_mapped_file_ ) /
      ( (double) bytes_per_line ) );
    const unsigned long int bytes_per_file = lines_per_file * 
      bytes_per_line;
    
    TEAGN_TRUE_OR_RETURN( 
      ( bytes_per_line <= max_bytes_per_mapped_file_ ),
      "The number of columns is too big" );
      
    const unsigned long int mapped_lines_nmb = total_lines_ -
      starting_line_index;
    const unsigned long int files_nmb = (unsigned long int)
      ceil(
        ( (double) mapped_lines_nmb ) / ( (double) lines_per_file )
      );
        
    /* Allocating lines2mm_objs_vector_ vector */
    
    unsigned long int curr_line_idx = 0;
    
    lines2mm_objs_vector_ = new MMObjsVectorNodePtrT[ total_lines_ ];
    
    TEAGN_TRUE_OR_RETURN( lines2mm_objs_vector_,
      "Memory allocation error" );
    
    for( curr_line_idx = 0 ; curr_line_idx < total_lines_ ; 
      ++curr_line_idx ) {
      
      lines2mm_objs_vector_[ curr_line_idx ] = 0;
    }      
      
    /* Allocating mapped memory lines */
    
    curr_line_idx = starting_line_index;
      
    for( unsigned long int curr_file = 0 ; curr_file < files_nmb ;
      ++curr_file ) {
      
      /* Creating the new mapped memory objets node */
      
      MMObjsVectorNodePtrT new_node_ptr = new MMObjsVectorNodeT;
      
      new_node_ptr->mm_ptr_ = new TeMappedMemory();
      TEAGN_TRUE_OR_RETURN( new_node_ptr->mm_ptr_->reset( 
        bytes_per_file, 
        false ), "Unable to allocate mapped memory file" )
        
      new_node_ptr->first_line_idx = curr_line_idx;
      new_node_ptr->lines_bound_ = curr_line_idx + lines_per_file;
      if( new_node_ptr->lines_bound_ > total_lines_ ) {
        new_node_ptr->lines_bound_ = total_lines_;
      }
      
      mapped_memory_objs_ptrs_vect_.push_back( new_node_ptr );
      
      while( curr_line_idx < new_node_ptr->lines_bound_ ) {
        lines2mm_objs_vector_[ curr_line_idx ] = new_node_ptr;
      
        ++curr_line_idx;
      }
    }
    
    return true;
  }  
  
  
  template< class T >
  T* TePDIMatrix< T >::scanLine( const unsigned int& line ) const
  {
    TEAGN_DEBUG_CONDITION( lines_pointers_vector_ != 0, 
      "Trying to access an empty matrix" );
    TEAGN_DEBUG_CONDITION( line < total_lines_, 
      "Trying to access an invalid line [" + Te2String( line ) + "]" );
      
    T* line_ptr = lines_pointers_vector_[ line ];
      
    if( line_ptr ) {
      return line_ptr;
    } else {
      /* Disable the current mapping */
      
      unsigned int curr_line = 0;
      unsigned int lines_bound = 0;
      
      if( current_active_mmobj_ptr_ ) {
        current_active_mmobj_ptr_->mm_ptr_->toggle( false );
        lines_bound = current_active_mmobj_ptr_->lines_bound_;
        
        for( curr_line = current_active_mmobj_ptr_->first_line_idx ; 
          curr_line < lines_bound ; ++curr_line ) {
          
          lines_pointers_vector_[ curr_line ] = 0;
        }
      }
      
      /* Enabling the required mapping */
      
      current_active_mmobj_ptr_ = lines2mm_objs_vector_[ line ];
      TEAGN_DEBUG_CONDITION( current_active_mmobj_ptr_,
        "Unable to find the required mapped memory object" )

      TEAGN_TRUE_OR_THROW( current_active_mmobj_ptr_->mm_ptr_->toggle( 
        true ), 
        "Unable to activate the required mapped memory object" )
        
      lines_bound = current_active_mmobj_ptr_->lines_bound_;
      T* first_line_ptr = 
        (T*)current_active_mmobj_ptr_->mm_ptr_->getPointer();
      TEAGN_DEBUG_CONDITION( first_line_ptr, 
        "Invalid line pointer returned from the mapped memory object" )
      
      for( curr_line = current_active_mmobj_ptr_->first_line_idx ; 
        curr_line < lines_bound ; ++curr_line ) {
        
        lines_pointers_vector_[ curr_line ] = first_line_ptr;
        
        first_line_ptr += total_columns_;
      }
      
      TEAGN_DEBUG_CONDITION( lines_pointers_vector_[ line ],
        "Invalid line pointer" );
      
      /* Returning the new mapped line */
      
      return lines_pointers_vector_[ line ];
    }
  } 
  
/** @example TePDIMatrix_test.cpp
 *    Shows how to use this class.
 */    
  
#endif //TEPDIMATRIX_HPP

