/*
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*/

/*! \file TePDIChainCode4.hpp
    This file contains the definition of a class to handle a chain code4
*/

#ifndef TEPDICHAINCODE4_HPP
#define TEPDICHAINCODE4_HPP

#include "TePDIDefines.hpp"
#include <TeGeometry.h>

/**
 * @brief Class ChainCode4 models a 2D (opened or closed) line with points sampled
 * over a rectangular grid. Line segments can be horizontal or vertical and
 * with size equal to the cell resolution. Only the first point is stored
 * (in both the image coordinate system and the projection coordinate system).
 * Each following point is represented by its relative position to the
 * preceeding point. These relative positions are: north, east, south, west.
 * For example, let a line in the image coord. system (line, column) be:
 * { (1,1),(1,2),(1,3),(2,3),(3,3),(3,2),(2,2) }. The chain coded version of
 * this line is: { (1,1), E, E, S, S, W, N }. This encoding approach is known
 * as chain-coding, or freeman-coding.
 */
class PDI_DLL TePDIChainCode4
{
public:

        /**
         * @brief Default constructor -
         * Creates a ChainCode4 with data buffer 128 bytes long.
         * If initialization failed (memory alocation) the object
         * is put in a bad state (Flag = 0).
         */
        TePDIChainCode4();

        /**
         * @brief Alternative constructor -
         * Initiaizes a ChainCode4 with data buffer "naco"
         * bytes long. If initialization failed (memory alocation)
         *
         * @param naco: the buffer size in bytes
         */
        TePDIChainCode4( int naco );

        /**
         * @brief Copy constructor -
         * If initialization failed (memory alocation)
         *
         * @param cha: an instance of ChainCode4
         */
        TePDIChainCode4( const TePDIChainCode4 &cha );

        /**
         * @brief Default destuctor.
         */
        ~TePDIChainCode4();

        /**
         * @brief Assignment operation  between ChainCode4 objects -
         * the object is put in a bad state (Flag = 0).
         *
         * @param chi: an instance of ChainCode4
         * @return an instance of ChainCode4
         */
        TePDIChainCode4& operator=( const TePDIChainCode4 &chi );

        /**
         * @brief Check if object is in a valid state.
         *
         * @return true if ok, otherwise false
         */
        bool isOK();

        /**
         * @brief Return the number of directions in the chain
         *
         * @return The number of directons
         */
        int     numDirections();

        /**
         * @brief Set the first point of the chain
         *
         * @param P: first point of the chain
         * @return true if ok, otherwise false
         */
        bool setFirstPoint( TeCoord2D P );

        /**
         * @brief Get the first point of the chain
         *
         * @param P: reference to the first point
         * @return true if ok, otherwise false
         */
        void getFirstPoint( TeCoord2D& P ) const;

        /**
         * @brief Set pixel size (both dimensions)
         *
         * @param rh: horizontal resolution
         * @param rv: vertical resolution
         * @return true if ok, otherwise false
         */
        bool setResolution( double rh, double rv );

        /**
         * @brief Get pixel size (both dimensions)
         *
         * @param rh: horizontal resolution
         * @param rv: vertical resolution
         */
        void    getResolution( double *rh, double *rv ) const;

        /**
         * @brief Add a direction at the end of chain
         *
         * @param d: direction value
         * @return true if ok, otherwise false
         */
        bool add( int d );

        /**
         * @brief Get a direction the direction at position "pos"
         *
         * @param pos: position to be extracted
         * @param d: reference to the direction value
         * @return true if ok, otherwise false
         */
        bool getDirection( int pos, int &d ) const;

private:
        /**
         * @brief Allocate data buffer
         *
         * @param nby:  buffer size in bytes
         * @return true if ok, otherwise false
         */
        bool allocBuffer( int nby );

        /**
         * @brief Free data buffer
         */
        void freeBuffer();

        /**
         * @brief Grows data buffer "Naco" bytes.
         *
         * @return true if ok, otherwise false
         */
        bool expandBuffer();

        /**
         * @brief Initialize ChainCode4 data members and alocate a data buffer
         * "naco" bytes long. If initialization failed (memory alocation)
         *
         * @param naco: the buffer size in bytes
         */
        void initialize( int naco );

        /** @brief Undocumented */
        bool Flag;              // 1 = living, 0 = dead
        
        /** @brief Undocumented */
        TeCoord2D firstPoint;
        
        /** @brief Horizontal dimension of pixel */
        double  Rh;
        
        /** @brief Vertical dimension of pixel */
        double  Rv;
        
        /** @brief Chain size in directions */
        int     NDir;
        
        /** @brief Chain size in bytes */
        int     NBytes;
        
        /** @brief Value used for chain growing */
        int     Naco;
        
        /** @brief Chain data */
        char    *Cadeia;

};

#endif
