/* $Id: lt_status.h,v 1.1 2006/10/27 11:40:40 lubia Exp $ */
/* //////////////////////////////////////////////////////////////////////////
//                                                                         //
// This code is Copyright (c) 2004 LizardTech, Inc, 1008 Western Avenue,   //
// Suite 200, Seattle, WA 98104.  Unauthorized use or distribution         //
// prohibited.  Access to and use of this code is permitted only under     //
// license from LizardTech, Inc.  Portions of the code are protected by    //
// US and foreign patents and other filings. All Rights Reserved.          //
//                                                                         //
////////////////////////////////////////////////////////////////////////// */
/* PUBLIC */

/**      
 * @file
 *
 * Declaration of the LT_STATUS type and some commonly used status codes.
 *
 * @note This file is C-callable.
 */

#ifndef LT_STATUS_H
#define LT_STATUS_H

#if defined(LT_COMPILER_MS)
   #pragma warning(push,4) 
#endif

/**
 * @typedef LT_STATUS
 *
 * An integral status code.
 *
 * This value is explicitly sized, i.e. lt_uint32 and not an "unsigned
 * long", so it can be passed safely between pre-compiled libraries.  Each
 * LizardTech project/library is assigned ranges of values, e.g. 1000-1999,
 * for project-specific, local usage.  Values 0-100 are general,
 * LizardTech-wide codes.  Values 65000-65535 are reserved for customer use.
 */
typedef lt_uint32 LT_STATUS;


/**
 * @name Common status codes
 */
/*@{*/

/**
 * status code indicating success
 *
 * This status code indicates the function completed sucessfully.
 */
#define LT_STS_Success ((LT_STATUS)0)

/**
 * status code indicating failure
 *
 * This status code indicates general, nonspecific failure.  Use sparingly;
 * project-specific codes are preferred where possible.
 */
#define LT_STS_Failure ((LT_STATUS)1)

/**
 * status code indicating unintialized variable
 *
 * This status code is only used as a sentinal value for initializing
 * LT_STATUS variables: it is the "-1" for status codes.  A conformant
 * function should _never_ return this value.
 */
#define LT_STS_Uninit ((LT_STATUS)2)
      
/**
 * status code indicating bad function parameter
 *
 * This status code indicates that one of the parameters passed to the
 * function was invalid, according to the documented interface rules.
 * Examples: an integer value is out of the allowed range, or a pointer
 * 
 */
#define LT_STS_BadParam ((LT_STATUS)3)
      
/**
 * status code indicating bad calling context for function
 *
 * This status code indicates that the function was called in some
 * unexpected fashion: not that there was anything wrong with the
 * parameters per se, it's just that the world was not set up for
 * you to make this particular call in this particular way right now --
 * you've violated a documented  execution constraint of some sort.
 * Examples: calling a class's member function before calling initialize(),
 * calling decode before setting a scene or an output buffer, etc.
 * (Contrast this with LT_STS_BadParam, which is a more specific
 * kind of problem.)
 */
#define LT_STS_BadContext ((LT_STATUS)4)

/**
 * status code indicating 3rd-party library error
 *
 * This status code indicates an unknown or otherwise undiagnosable
 * exception or error was generated by a call into a 3rd party library.
 */
#define LT_STS_ForeignError ((LT_STATUS)5)
      
/**
 * status code indicating unreachable code
 *
 * This status code is used for code that by definition should not or
 * cannot be reached, e.g. the default case in a switch statement for
 * which you think you've covered the full range of legal case values,
 * where you're left with a dangling return statement.  A return of
 * LT_STS_NotReached is often preceeded by "LT_ASSERT(false)".
 */
#define LT_STS_NotReached ((LT_STATUS)6)
      
/**
 * status code indicating bad NULL pointer dereference
 *
 * This status code is used when checking a pointer to make sure that
 * it is not NULL, prior to dereferencing it.
 */
#define LT_STS_NullPointer ((LT_STATUS)7)

/**
 * status code indicating new/malloc/calloc failed
 *
 * This status code is used when allocating memory returns a NULL 
 * pointer.
 */
#define LT_STS_OutOfMemory ((LT_STATUS)8)

/*@}*/
      

/**
 * @name Macros for checking status codes
 */
/*@{*/

/** checks return value for success status */
#define LT_SUCCESS( err ) ((err)==LT_STS_Success)
/** checks return value for failure status */
#define LT_FAILURE( err ) ((err)!=LT_STS_Success)

/*@}*/


#if defined(LT_COMPILER_MS)
   #pragma warning(pop) 
#endif

#endif /* LT_STATUS_H */
