/*
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*/

/*! \file TePDI2DTree.hpp
    This file contains the definition of a classes to handle a 2Dtree
*/

#ifndef TEPDI2DTREE_HPP
#define TEPDI2DTREE_HPP

#include <vector>
#include "TePDIDefines.hpp"

/**
 * @brief Class TePDI2DTree
 * Very simple implementation of a 2DTree with 3 methods :
 *   clear(.), addObject(.), isPresent(.)
 * the two template O and C are :
 *    O : class that contain everything you want but a public access to
 *        a Coordinate of type C called coord_ that is used to store the object O in the 2dtree
 *    C : class that contains the cartesian coordinate system (x,y)
 *        x accessible by the method x()
 *        y accessible by the method y()
 */
template <class O, class C>
class PDI_DLL TePDI2DTree {

public:

        /**
          * @brief default constructor.
          */
        TePDI2DTree(){};

        /**
         * @brief alternativge constructor.
         * @param Obj Undocumented.
         * @param d Undocumented.
         * @param onX Undocumented.
         */
        TePDI2DTree(O* Obj, int d, bool onX):ptRight_(NULL), ptLeft_(NULL), isOnX_(onX),
                depth_(d), root_(Obj){
                v_ = new vector<O*>;
                v_->push_back(Obj);
        };

        /**
         * @brief clean the memory.
         */
        void clear(){
                if (ptRight_) ptRight_->clear();
                ptRight_=NULL;
                if (ptLeft_)  ptLeft_->clear();
                ptLeft_=NULL;
                v_->clear();
                return;
        };

        /**
         * @brief default destructor.
         */
        ~TePDI2DTree(){
                this->clear();
        };

    /**
     * @brief Add object.
     * @param Obj Object pointer.
     * @return Undocumented.
     */
    bool addObject(O* Obj){
                TePDI2DTree* curTree=this;
                O* root;
                bool isOnX;

                while (true) {
                        if (depth_==0) {
                                v_->push_back(Obj);
                                return true;
                        }
                        root=curTree->getRoot();
                        isOnX=curTree->onX();
                        if (isOnX){
                                if (Obj->coord_.x() < root->coord_.x() ) {
                                        if (curTree->getLeftTree()==NULL){
                                                curTree->setLeftTree(new TePDI2DTree(Obj, depth_-1, !isOnX));
                                                break;
                                        }
                                        else
                                                curTree=curTree->getLeftTree();
                                } else if (Obj->coord_.x() > root->coord_.x() ) {
                                        if (curTree->getRightTree()==NULL){
                                                curTree->setRightTree(new TePDI2DTree(Obj, depth_-1, !isOnX));
                                                break;
                                        } else
                                                curTree= curTree->getRightTree();
                                } else {
                                        curTree->getVector()->push_back(Obj);
                                        break;
                                }
                        } else {
                                if (Obj->coord_.y() < root->coord_.y() ) {
                                        if (curTree->getLeftTree()==NULL){
                                                curTree->setLeftTree(new TePDI2DTree(Obj, depth_-1, !isOnX));
                                                break;
                                        }
                                        else
                                                curTree=curTree->getLeftTree();

                                } else if (Obj->coord_.y() > root->coord_.y() ) {
                                        if (curTree->getRightTree() == NULL)
                                                curTree->setRightTree( new TePDI2DTree(Obj, depth_-1, !isOnX));
                                        else
                                                curTree=curTree->getRightTree();
                                } else {
                                        curTree->getVector()->push_back(Obj);
                                        break;
                                }
                        }
                }
                return true;
        }

        /**
         * Check if there is an object O with the coordinate P
         *
         * @param P coordinate of the object
         * @return pointer to that object if present otherwise NULL
         */
    O* isPresent(C P) {
                TePDI2DTree* curTree=this;
                vector<O*> *v;
                C coordRoot;

                while (curTree != NULL) {
                        coordRoot=curTree->getRoot()->coord_;

                        if (P == coordRoot)
                                return curTree->getRoot();

                        if (curTree->getDepth() == 0){
                                v=curTree->getVector();
                                for (unsigned int i=0 ; i < v->size() ; i++){
                                        if (v->operator [](i)->coord_==P)
                                                return v->operator [](i);
                                }
                                return false;
                        }

                        if (curTree->onX()) {

                                if (P.x() <coordRoot.x()) {
                                                curTree=curTree->getLeftTree();
                                                continue;
                                        }
                                else if (P.x() > coordRoot.x()) {
                                                curTree=curTree->getRightTree();
                                                continue;
                                        }
                                else {
                                        v=curTree->getVector();
                                        for (unsigned int i=0 ; i < v->size() ; i++){
                                                if (v->operator [](i)->coord_==P)
                                                        return v->operator [](i);
                                        }
                                        return NULL;
                                }
                        } else {
                                if (P.y() < coordRoot.y()) {
                                                curTree=curTree->getLeftTree();
                                                continue;
                                        }
                                else if (P.y() > coordRoot.y()) {
                                                curTree=curTree->getRightTree();
                                                continue;
                                        }
                                else {
                                        v=curTree->getVector();
                                        for (unsigned int i=0 ; i< v->size() ; i++){
                                                if (v->operator [](i)->coord_==P)
                                                        return v->operator [](i);
                                        }
                                        return NULL;
                                }

                        }

                }
                return NULL;
        }

private:
        /**
         * @brief get the root.
         * @return The root.
         */
        O* getRoot()
                {return root_;}
                
        /**
         * @brief get the left tree.
         * @return the left tree.
         */
        TePDI2DTree* getLeftTree()
                {return ptLeft_;}
                
        /**
         * @brief get the right tree.
         * @return the right tree.
         */
        TePDI2DTree* getRightTree()
                {return ptRight_;}
                
        /**
         * @brief set the left tree
         * @param t Undocumented.
         */
        void setLeftTree(TePDI2DTree* t)
                {ptLeft_=t;}
                
        /**
         * @brief set the right tree.
         * @param t Undocumented.
         */
        void setRightTree(TePDI2DTree* t)
                {ptRight_=t;}
                
        /**
         * @brief get the max depth of the tree.
         * @return max depth.
         */
        int getDepth()
                {return depth_;}
                
        /**
         * @brief get the vector of coordinates associated to the tree.
         * @return the vector.
         */
        vector<O*>* getVector()
                {return v_;}
                
        /**
         * @brief return true if the onX.
         * @return true if the onX.
         */
        bool onX()
                {return isOnX_;}

        /** @brief Undocumented. */        
        TePDI2DTree* ptRight_;
        
        /** @brief Undocumented. */        
        TePDI2DTree* ptLeft_;
        
        /** @brief Undocumented. */        
        bool isOnX_;
        
        /** @brief Undocumented. */        
        int depth_;
        
        /** @brief Undocumented. */        
        O* root_;
        
        /** @brief Undocumented. */        
        vector<O*> *v_;
};

#endif
